<?php
/*--------------------------------------------------------------
   TitleCollection.php 2021-06-02
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections;

use ArrayIterator;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Features\HasLanguageCode;
use IteratorAggregate;
use Webmozart\Assert\Assert;

abstract class TitleCollection implements IteratorAggregate
{
    private const GERMAN_LANGUAGE_CODE = "de";
    
    private const ENGLISH_LANGUAGE_CODE = "en";
    
    /**
     * @var array
     */
    private $values;
    
    
    /**
     * Constructor.
     *
     * @param HasLanguageCode[] $values
     */
    protected function __construct(array $values)
    {
        $hasGermanTitle  = false;
        $hasEnglishTitle = false;
        
        $this->values = [];
        
        foreach ($values as $value) {
            $languageCode = $value->languageCode()->value();
            
            if (!$hasGermanTitle && $languageCode === self::GERMAN_LANGUAGE_CODE) {
                $hasGermanTitle = true;
            }
            
            if (!$hasEnglishTitle && $languageCode === self::ENGLISH_LANGUAGE_CODE) {
                $hasEnglishTitle = true;
            }
            
            $this->values[$languageCode] = $value;
        }
        
        Assert::true($hasGermanTitle);
        Assert::true($hasEnglishTitle);
    }
    
    
    /**
     * Return item by language code.
     */
    public function getByLanguageCode(string $languageCode): HasLanguageCode
    {
        Assert::stringNotEmpty($languageCode);
        
        return $this->values[$languageCode] ?? $this->values[self::GERMAN_LANGUAGE_CODE];
    }
    
    
    /**
     * @inheritDoc
     */
    public function getIterator(): ArrayIterator
    {
        return new ArrayIterator(array_values($this->values));
    }
}
